module net.BurtonRadons.dig.common.brush;

private import net.BurtonRadons.dig.platform.base;

/** Rotate around a point. */
class RotateContorter
{
public:
    /** Registers the eval method with the contorter. */
    this (Contorter *con)
    {
        con.add (&eval);
    }

    /** Set the rotation angle to a radian. */
    void angleRad (float value)
    {
        digCommonAngle = value;
        digCommonSinAngle = sin (digCommonAngle);
        digCommonCosAngle = cos (digCommonAngle);
    }

    /** Set the rotation angle to a degree. */
    void angleDeg (float value)
    {
        angleRad (value * PI / 180.0);
    }

    /** Set the center of rotation. */
    void center (float x, float y)
    {
        digCommonCX = x;
        digCommonCY = y;
    }

    /** Rotate the point around the center; this is registered with the Contorter. */
    void eval (inout float x, inout float y)
    {
        float rx = x - digCommonCX;
        float ry = y - digCommonCY;

        x = digCommonSinAngle * rx + digCommonCosAngle * ry + digCommonCX;
        y = digCommonCosAngle * rx - digCommonSinAngle * ry + digCommonCY;
    }

/+
#ifdef DoxygenShouldSkipThis
+/

    float digCommonCX = 0; /**< Horizontal center to rotate around. */
    float digCommonCY = 0; /**< Vertical center to rotate around. */
    float digCommonAngle = 0; /**< Angle in radians to rotate. */
    float digCommonSinAngle; /**< sin (angle). */
    float digCommonCosAngle; /**< cos (angle). */
    
/+
#endif
+/
}

/** Transform with a matrix.  The response of this when applied to bitmap
  * points may be confusing.  Think of it this way: if you scale a coordinate
  * by 10%, that makes the bitmap ten times larger by focusing on a smaller
  * area.
  */

class TransformContorter
{
    /** Registers the eval method with the contorter. */
    this (Contorter *con)
    {
        con.add (&eval);
    }

    /** Resets the matrix to identity. */
    void identity ()
    {
        digCommonXX = 1;
        digCommonXY = 0;
        digCommonXP = 0;
        digCommonYX = 0;
        digCommonYY = 1;
        digCommonYP = 0;
    }

    /** Scale the matrix by an amount. */
    void scale (float x, float y)
    {
        multiply (x, 0, 0, 0, y, 0);
    }

    /** Scale the matrix by an inverse amount. */
    void blowup (float x, float y)
    {
        multiply (1 / x, 0, 0, 0, 1 / y, 0);
    }

    /** Translate the matrix by an amount. */
    void translate (float x, float y)
    {
        multiply (1, 0, x, 0, 1, y);
    }

    /** Rotate the matrix around (0, 0) using degrees. */
    void rotate (float angle)
    {
        rotate (angle, 0, 0);
    }

    /** Rotate the matrix around a specific point using degrees. */
    void rotate (float angle, float x, float y)
    {
        float c = cos (angle * PI / 180);
        float s = sin (angle * PI / 180);

        translate (-x, -y);
        multiply (c, -s, 0,
                  s, c, 0);
        translate (x, y);
    }

    /** Multiply the matrix by another matrix. */
    void multiply (float mxx, float mxy, float mxp, float myx, float myy, float myp)
    {
        float lxx = digCommonXX, lxy = digCommonXY, lxp = digCommonXP;
        float lyx = digCommonYX, lyy = digCommonYY, lyp = digCommonYP;

        digCommonXX = (lxx * mxx) + (lyx * mxy);
        digCommonXY = (lxy * mxx) + (lyy * mxy);
        digCommonXP = (lxp * mxx) + (lyp * mxy) + mxp;

        digCommonYX = (lxx * myx) + (lyx * myy);
        digCommonYY = (lxy * myx) + (lyy * myy);
        digCommonYP = (lxp * myx) + (lyp * myy) + myp;
    }

    /** Apply the matrix to the point. */
    void eval (inout float x, inout float y)
    {
        float rx = x, ry = y;

        x = rx * digCommonXX + ry * digCommonXY + digCommonXP;
        y = rx * digCommonYX + ry * digCommonYY + digCommonYP;
    }

/+
#ifdef DoxygenShouldSkipThis
+/

    float digCommonXX = 1; /**< Multiplier for the x coordinate against x. */
    float digCommonXY = 0; /**< Multiplier for the x coordinate against y. */
    float digCommonXP = 0; /**< Constant added for the x coordinate. */
    float digCommonYX = 0; /**< Multiplier for the y coordinate against x. */
    float digCommonYY = 1; /**< Multiplier for the y coordinate against y. */
    float digCommonYP = 0; /**< Constant added for the y coordinate. */
    
/+
#endif
+/
}

/** Perturb according to a sin wave. */
class SwimContorter
{
    /** Register the eval method with the contorter. */
    this (Contorter *con)
    {
        con.add (&eval);
    }

    /** Subtract the center and divide by scale. */
    void eval (inout float x, inout float y)
    {
        float rx = (x - digCommonCX) / digCommonSX;
        float ry = (y - digCommonCY) / digCommonSY;

        x = x + sin (ry) * digCommonMX;
        y = y + sin (rx) * digCommonMY;
    }

    /** Set the center, subtracted from the point before apply. */
    void center (float x, float y) { digCommonCX = x; digCommonCY = y; }

    /** Set the scale, dividing the point after subtracting center. */
    void scale (float x, float y) { digCommonSX = x; digCommonSY = y; }

    /** Set the multiplication factor for the sin wave. */
    void multiply (float x, float y) { digCommonMX = x; digCommonMY = y; }

/+
#ifdef DoxygenShouldSkipThis
+/

    float digCommonCX = 0; /**< Horizontal offset to the center of the swim. */
    float digCommonCY = 0; /**< Vertical offset to the center of the swim. */
    float digCommonSX = 10; /**< Size of a single wave, horizontally. */
    float digCommonSY = 10; /**< Size of a single wave, vertically. */
    float digCommonMX = 5; /**< Horizontal amount of distortion. */
    float digCommonMY = 5; /**< Vertical amount of distortion. */
    
/+
#endif
+/
}

/** A list of delegates that takes a point and contorts it. */
struct Contorter
{
/+
#ifndef DOXYGEN_SHOULD_SKIP_THIS
+/
    typedef void delegate (inout float x, inout float y) ContorterFunc;
/+
#endif
+/

    /** The list of contorting delegates, appended to by add.
      * These are applied in the given order.  You can modify this
      * array however you want.
      */

    ContorterFunc [] contorters;

    /** Add a contorter function. */
    void add (ContorterFunc func)
    {
        contorters ~= func;
    }

    /** Delete the contorters list and reset it. */
    void empty ()
    {
        delete contorters;
        contorters = null;
    }

    /** Apply each contorter in order of addition. */
    void eval (inout float x, inout float y)
    {
        ContorterFunc *c = contorters;
        ContorterFunc *e = c + contorters.length;

        for ( ; c < e; c ++)
            (*c) (x, y);
    }
}

/** A brush returns a color from a given point.  Brushes also hold pen
  * information (probably not forever, as it's not a good match).  This is
  * primarily used by the @a Bitmap class.
  */

class Brush
{
    Contorter contort; /**< The contortion to apply to points before sampling the color. */

    float width; /**< When a pen, the width of it in pixels */

    /** Evaluate the brush at a point. */
    Color eval (float x, float y) { return Color.Black; }
}

/** This brush returns a single color from its evaluator. */
class BrushColor : Brush
{
    /** Set the color that eval will return. */
    this (Color col) { this.digCommonColor = col; }

    /** Return the color passed during construction. */
    Color eval (float x, float y) { return digCommonColor; }

    /** Assign the color. */
    void color (Color value) { digCommonColor = value; }

/+
#ifdef DoxygenShouldSkipThis
+/

    Color digCommonColor; /* The color of the brush */
    
/+
#endif
+/
}

/** A brush that returns samples from a bitmap from its evaluator. */
class BrushBitmap : Brush
{
    private import net.BurtonRadons.dig.platform.bitmap;
    
    /** Set the bitmap that will be sampled from.  Wrap mode is wrap. */
    this (Bitmap bitmap)
    {
        digCommonBitmap = bitmap;
    }

    /** Set the center subtracted from the sample point. */
    void center (float x, float y)
    {
        digCommonCX = x;
        digCommonCY = y;
    }

    /** Evaluate the bitmap at that point.  First it divides by scale and 
      * subtracts the @a center.  Then it applies the contortion; then it
      * wraps and returns the color sampled at that point.
      */

    Color eval (float x, float y)
    {
        return evalNearest (x, y);
    }

    /** Evaluate using no interpolation. */

    final Color evalNearest (float x, float y)
    {
        int w = digCommonBitmap.width ();
        int h = digCommonBitmap.height ();

        x = x / digCommonSX - digCommonCX;
        y = y / digCommonSY - digCommonCY;
        contort.eval (x, y);

        switch (digCommonEdges)
        {
            case 0:
                x = ((x % w) + w) % w;
                y = ((y % h) + h) % h;
                break;

            case 1:
                x = x < 0 ? 0 : x > w - 1 ? w - 1 : x;
                y = y < 0 ? 0 : y > h - 1 ? h - 1 : y;
                break;

            case 2:
                if (x < 0 || y < 0 || x >= w || y >= h)
                    return digCommonColor;
                break;
        }

        return digCommonBitmap.get (x, y);
    }

    /** Evaluate using bilinear interpolation. */

    final Color evalBilinear (float x, float y)
    {
        int w = digCommonBitmap.width ();
        int h = digCommonBitmap.height ();
        Color a, b, c, d;
        int nx, ny;

        nx = x;
        ny = y;
        x = x / digCommonSX - digCommonCX;
        y = y / digCommonSY - digCommonCY;
        contort.eval (x, y);

        switch (digCommonEdges)
        {
            case 0:
                x = ((x % w) + w) % w;
                y = ((y % h) + h) % h;
                nx = (x + 1) % w;
                ny = (y + 1) % h;
                break;

            case 1:
                x = x < 0 ? 0 : x > w - 1 ? w - 1 : x;
                y = y < 0 ? 0 : y > h - 1 ? h - 1 : y;
                nx = (x == w - 1 ? x : x + 1);
                ny = (y == h - 1 ? y : y + 1);
                break;

            case 2:
                if (x < -1 || x > w || y < -1 || y > h)
                    return digCommonColor;
                if (x >= 0 && y >= 0 && x < w - 1 && y < h - 1)
                {
                    nx = x + 1;
                    ny = y + 1;
                }
                else
                {
                    if (x < 0)
                    {
                        if (y < 0)
                        {
                            a = b = c = digCommonColor;
                            d = digCommonBitmap.get (0, 0);
                            goto skipColors;
                        }
                        else if (y >= h - 1)
                        {
                            a = c = d = digCommonColor;
                            d = digCommonBitmap.get (0, h - 1);
                        }
                        else
                        {
                            a = c = digCommonColor;
                            b = digCommonBitmap.get (0, y);
                            d = digCommonBitmap.get (0, y + 1);
                        }
                    }
                    else if (x >= w - 1)
                    {
                        if (y < 0)
                        {
                            a = b = d = digCommonColor;
                            c = digCommonBitmap.get (w - 1, 0);
                        }
                        else if (y >= h - 1)
                        {
                            b = c = d = digCommonColor;
                            a = digCommonBitmap.get (w - 1, h - 1);
                        }
                        else
                        {
                            b = d = digCommonColor;
                            a = digCommonBitmap.get (w - 1, y);
                            c = digCommonBitmap.get (w - 1, y + 1);
                        }
                    }
                    else
                    {
                        if (y < 0)
                        {
                            a = b = digCommonColor;
                            c = digCommonBitmap.get (x, 0);
                            d = digCommonBitmap.get (x + 1, 0);
                        }
                        else if (y >= h - 1)
                        {
                            c = d = digCommonColor;
                            a = digCommonBitmap.get (x, h - 1);
                            b = digCommonBitmap.get (x + 1, h - 1);
                        }
                        else
                        {
                            a = digCommonBitmap.get (x, y);
                            b = digCommonBitmap.get (x + 1, y);
                            c = digCommonBitmap.get (x, y + 1);
                            d = digCommonBitmap.get (x + 1, y + 1);
                        }
                    }

                    goto skipColors;
                }

                break;
        }

        a = digCommonBitmap.getx (x, y);
        b = digCommonBitmap.getx (nx, y);
        c = digCommonBitmap.getx (x, ny);
        d = digCommonBitmap.getx (nx, ny);
    skipColors:

        return a.blend2 (b, c, d, (x % 1.0) * 255, (y % 1.0) * 255);
    }

    /** Set wrapping mode to wrap.  When a sample point is out-of-range,
      * it wraps around to the other side of the bitmap.
      */

    void wrap () { digCommonEdges = 0; }

    /** Set wrapping mode to saturate.  When a sample point is out-of-range,
      * the nearest in-range edge is used for the color.
      */

    void saturate () { digCommonEdges = 1; }

    /** Set wrapping mode to border.  When a sample point is out-of-range,
      * the passed color is returned.
      */

    void border (Color color) { digCommonColor = color; digCommonEdges = 2; }

/+
#ifdef DoxygenShouldSkipThis
+/

    Bitmap digCommonBitmap;
    float digCommonCX = 0, digCommonCY = 0;
    float digCommonSX = 1, digCommonSY = 1;
    int digCommonEdges = 0;
    Color digCommonColor;
    
/+
#endif
+/
}

/** A blending brush that takes any number of input brushes and
  * layers them together.
  */

class BrushBlend : Brush
{
    /** The available blending modes that determines the blending color
      * from the current (a) and layer (b) colors.
      */

    enum Blend
    {
        Normal = 0,     //!< Normal blending (b).
        Multiply = 1,   //!< Multiply the colors together (a * b).
        Add = 2,        //!< Add the colors together (a + b).
        Subtract = 3,   //!< Subtract the colors (a - b).
        Difference = 4, //!< Get the difference between the colors (abs (a - b)).
        Darken = 5,     //!< Get the minimum intensity of each channel (min (a, b)).
        Lighten = 6,    //!< Get the maximum intensity of each channel (max (a, b)).
        Hue = 7,        //!< Use the hue from the layer color (a with hue from b).
        Saturation = 8, //!< Use the saturation from the layer color (a with saturation from b).
        Color = 9,      //!< Use the hue and saturation from the layer color (a with hue and saturation from b).
        Luminance = 10, //!< Use the luminance/value from the layer color (a with luminance from b).
    }

    /** An individual layer structure. */

    struct Layer
    {
        Brush brush; /**< The brush to sample colors from. */
        Blend blend; /**< The blending mode to use to get the blending color. */
        ubyte alpha; /**< The layer opacity, from 0 (transparent) to 255 (opaque). */

        /** Sample the layer using pre-contorted points.  Multiplies the
          * opacity in the color with the layer opacity.
          */

        Color eval (float x, float y)
        {
            Color c = brush.eval (x, y);

            return AColor (c.r, c.g, c.b, c.a * alpha / 255);
        }
    };

    /** The list of layers in the brush, modified using add.
      * You can modify this list to your pleasure; however,
      * pointers to it will be invalidated by an add call.
      */

    Layer [] layers;

    /** Add an opaque layer to the brush using normal blend mode. */
    void add (Brush brush) { opAdd (brush, Blend.Normal, 255); }

    /** Add an opaque layer to the brush. */
    void add (Brush brush, Blend blend) { opAdd (brush, blend, 255); }

    /** Add a layer to the brush. */
    void add (Brush brush, Blend blend, ubyte alpha)
    {
        Layer me;

        me.brush = brush;
        me.blend = blend;
        me.alpha = alpha;

        layers ~= me;
    }

    /** Cycle through the layers on the brush, sampling at each point
      * and blending the colors together.  The blending mode of the
      * first layer is ignored (but its opacity is not).
      */

    Color eval (float x, float y)
    {
        Layer *l = layers;
        Layer *e = l + layers.length;
        Color c, d;

        if (layers.length == 0)
            return Color.Black;

        contort.eval (x, y);
        c = l.eval (x, y);
        l ++;

        while (l < e)
        {
            d = l.eval (x, y);
            c = blendColor (c, d, l.blend);
            l ++;
        }

        return c;
    }

    /** Get the blending color from the current color (a),
      * the layer color (b), and the blending mode (m).
      */

    static Color blendColor (Color a, Color b, Blend m)
    {
        int ar = a.r, ag = a.g, ab = a.b, aa = a.a;
        int br = b.r, bg = b.g, bb = b.b, ba = b.a;

        switch (m)
        {
            case Blend.Normal: return a.blend (b);
            case Blend.Multiply: return a.blend (AColor (ar * br / 255, ag * bg / 255, ab * bb / 255, ba));
            case Blend.Add: return a.blend (SColor (ar - br, ag - bg, ab - bb, ba));
            case Blend.Subtract: return a.blend (SColor (ar - br, ag - bg, ab - bb, ba));
            case Blend.Difference: return a.blend (SColor (iabs (ar - br), iabs (ag - bg), iabs (ab - bb), ba));
            case Blend.Darken: return a.blend (SColor (imin (ar, br), imin (ag, bg), imin (ab, bb), ba));
            case Blend.Lighten: return a.blend (SColor (imax (ar, br), imax (ag, bg), imax (ab, bb), ba));
            case Blend.Hue:
            {
                float ah, as, av;
                float bh, bs, bv;

                a.toHSV (ah, as, av);
                b.toHSV (bh, bs, bv);
                return a.blend (Color.fromHSV (bh, as, av, b.a));
            }

            case Blend.Saturation:
            {
                float ah, as, av;
                float bh, bs, bv;

                a.toHSV (ah, as, av);
                b.toHSV (bh, bs, bv);
                return a.blend (Color.fromHSV (ah, bs, av, b.a));
            }

            case Blend.Color:
            {
                float ah, as, av;
                float bh, bs, bv;

                a.toHSV (ah, as, av);
                b.toHSV (bh, bs, bv);
                return a.blend (Color.fromHSV (bh, bs, av, b.a));
            }

            case Blend.Luminance:
            {
                float ah, as, av;
                float bh, bs, bv;

                a.toHSV (ah, as, av);
                b.toHSV (bh, bs, bv);
                return a.blend (Color.fromHSV (ah, as, bv, b.a));
            }
        }
    }
}

/** A linear or radial gradient */
class BrushGradient : public Brush
{
    /** Set the center of the gradient, subtracted from the sampling point. */
    void center (float x, float y) { digCommonCX = x; digCommonCY = y; }

    /** The sampling point is divided by this value after subtracting the center. */
    void scale (float value) { digCommonScale = value; }

    /** Add a point to the gradient.  Nominally p should be between 0 and 1;
      * however, it can be outside of that range.  Colors are linearly
      * interpolated between.
      */

    void add (float p, Color col)
    {
        for (int c; c < digCommonPoints.length; c ++)
        {
            if (p > digCommonPoints [c])
            {
                digCommonPoints.length = digCommonPoints.length + 1;
                for (int d = digCommonPoints.length - 1; d > c + 1; d --)
                    digCommonPoints [d] = digCommonPoints [d - 1];
                digCommonPoints [c + 1] = p;

                digCommonColors.length = digCommonColors.length + 1;
                for (int d = digCommonColors.length - 1; d > c + 1; d --)
                    digCommonColors [d] = digCommonColors [d - 1];
                digCommonColors [c + 1] = col;
                digCommonRecalculate ();
                return;
            }
        }

        digCommonPoints ~= p;
        digCommonColors ~= col;
        digCommonRecalculate ();
    }

    /** Set a previously defined point on the gradient or add it if this point
      * has not been defined.
      */

    void set (float p, Color col)
    {
        for (int c; c < digCommonPoints.length; c ++)
            if (digCommonPoints [c] == p)
            {
                digCommonColors [c] = col;
                digCommonRecalculate ();
                return;
            }

        opAdd (p, col);
    }

    /** Evaluate the gradient.  This first applies the contortion, then
      * subtracts the center and divides by scale.  If the gradient is
      * linear, then this calls at with the resultant y coordinate.  If
      * the gradient is radial, this calls at with the distance of x and
      * y from (0, 0).
      */

    Color eval (float x, float y)
    {
        float p;

        contort.eval (x, y);
        if (digCommonRadial)
            p = sqrt ((y - digCommonCY) * (y - digCommonCY) + (x - digCommonCX) * (x - digCommonCX)) / digCommonScale;
        else
            p = (y - digCommonCY) / digCommonScale;

        p = (p - digCommonLowQuick) * digCommonDeltaQuick;
        if (p < 0) return digCommonQuick [0];
        if (p >= digCommonQuickCount - 1) return digCommonQuick [digCommonQuickCount - 1];
        return digCommonQuick [p].blend (digCommonQuick [p + 1], p - (int) p);
    }

    /** Sample the gradient at a given point.  If it is out-of-range, it
      * returns the first or last sample point.  Otherwise it linearly
      * interpolates between the two nearest colors.
      */

    Color at (float t)
    {
        if (t < digCommonPoints [0]) return digCommonColors [0];
        if (t >= digCommonPoints [digCommonPoints.length - 1]) return digCommonColors [digCommonColors.length - 1];
        for (int c = 1; c < digCommonPoints.length; c ++)
        {
            float a = digCommonPoints [c - 1];
            float b = digCommonPoints [c];

            if (b >= t)
                return digCommonColors [c - 1].blend (digCommonColors [c], (t - a) / (b - a) * 255);
        }
    }

    /** Set radial gradient mode (true) or linear gradient mode (false).
      */

    void radial (bit value) { digCommonRadial = value; }

/+
#ifdef DoxygenShouldSkipThis
+/

    float digCommonCX = 0, digCommonCY = 0; /* Center */
    float digCommonScale = 1; /* Scale */
    Color [] digCommonColors;
    float [] digCommonPoints;
    bit digCommonRadial = false;

    const int digCommonQuickCount = 256;
    Color [digCommonQuickCount] digCommonQuick;
    float digCommonLowQuick;
    float digCommonHighQuick;
    float digCommonDeltaQuick; /* (highQuick - lowQuick) * quickCount */

    void digCommonRecalculate ()
    {
        digCommonLowQuick = digCommonPoints [0];
        digCommonHighQuick = digCommonPoints [digCommonPoints.length - 1];
        digCommonDeltaQuick = digCommonQuickCount / (digCommonHighQuick - digCommonLowQuick);
        for (int c; c < digCommonQuickCount; c ++)
            digCommonQuick [c] = at (digCommonLowQuick + c * (digCommonHighQuick - digCommonLowQuick) / digCommonQuickCount);
    }
    
/+
#endif
+/
}

/** Modulate the input by random noise. */
class BrushNoise : Brush
{
    /** Set the input brush. */
    this (Brush input) { digCommonInput = input; }

    /** The scale parameter; how far it can shift the color.
      * A scale of 5 can shift it up 2.5 or down 2.5.
      */

    void scale (float value)
    {
        digCommonScale = value;
        digCommonScaleHalf = value / 2;
    }

    /** Whether the noise is component-independent (false) or grayscale (true). */
    void uniform (bit value) { digCommonUniform = value; }

    /** Offset to thh coordinate. */
    void offset (float x, float y) { digCommonOffsetX = x; digCommonOffsetY = y; }

    /** Apply the contortion, sample the input, then modulate the color. */
    Color eval (float x, float y)
    {
        if (digCommonInput === null)
            return Color.Black;

        x += digCommonOffsetX;
        y += digCommonOffsetY;
        contort.eval (x, y);

        Color c = digCommonInput.eval (x, y);

        if (digCommonScale == 0) return c;

        int r = c.r, g = c.g, b = c.b;
        int cr, cg, cb;

        rand_seed (x > 0 ? x : -x + 640, y > 0 ? y : -y + 4328);
        if (digCommonUniform)
            cr = cg = cb = (rand () % digCommonScale) - digCommonScaleHalf;
        else
        {
            cr = (rand () % digCommonScale) - digCommonScaleHalf;
            cg = (rand () % digCommonScale) - digCommonScaleHalf;
            cb = (rand () % digCommonScale) - digCommonScaleHalf;
        }

        Color result = SColor (r + cr, g + cg, b + cb, c.a);
        return result;
    }

/+
#ifdef DoxygenShouldSkipThis
+/

    Brush digCommonInput; /* Input brush */
    float digCommonScale = 0; /* Scale */
    float digCommonScaleHalf = 0;
    bit digCommonUniform = false; /* Uniform bit */
    float digCommonOffsetX = 0, digCommonOffsetY = 0; /* Offset */
    
/+
#endif
+/
}

/** Apply a convolution filter to an input brush.  The filter is a matrix that
  * is applied to a set of samples from the input, with each sample multiplied
  * by an amount, and all the samples added together.  Then the result is
  * divided by an amount and that color is returned.
  */

class BrushFilter : Brush
{
    /** Set the input brush. */
    void input (Brush value) { digCommonInput = value; }

    /** Set the divisor.  When the divisor is not set, it is set according to
      * the sum of the cells in the matrix.  You can restore this property by
      * setting the divisor to float.nan.
      */

    void divisor (float value)
    {
        if (isnan (value))
        {
            digCommonUseDivisor = false;
            findDivisor ();
        }
        else
        {
            digCommonDivisor = value;
            digCommonUseDivisor = true;
        }
    }

    /** Get the set divisor or compute it. */

    float findDivisor ()
    {
        if (digCommonUseDivisor)
            return digCommonDivisor;
        digCommonDivisor = 0;
        for (float *c = digCommonMatrix, ce = c + digCommonMatrix.length; c < ce; c ++)
            digCommonDivisor += *c;
        return digCommonDivisor;
    }

    /** Resize the square matrix.  This clears all cells to zero.
      */

    void resize (int value)
    {
        delete digCommonMatrix;
        digCommonMatrix = new float [value * value];
        digCommonSize = value;
        digCommonMatrix [] = 0;
        findDivisor ();
    }

    /** Set the center of the filter matrix.  For example, (1, 1) on a
      * 3x3 filter makes the (1, 1) coordinate the input point.
      */

    void center (int x, int y) { digCommonCX = x; digCommonCY = y; }

    /** Set a cell on the filter. */

    void set (int x, int y, float v)
    {
        if (x < 0 || y < 0 || x >= digCommonSize || y >= digCommonSize)
            return;
        digCommonMatrix [x + y * digCommonSize] = v;
        findDivisor ();
    }

    /** Set to a line edge filter. */

    void setLineEdge ()
    {
        resize (3);
        center (1, 1);
        divisor (float.nan);
        set (1, 0, -1);
        set (0, 1, -1);
        set (2, 1, -1);
        set (1, 2, -1);
        set (1, 1, 5);
    }

    /** Set to a high pass filter. */

    void setHighPass (float scale)
    {
        resize (3);
        center (1, 1);
        divisor (float.nan);
        set (0, 0,  1); set (1, 0, -2);        set (2, 0,  1);
        set (0, 1, -2); set (1, 1, 6 * scale); set (2, 1, -2);
        set (0, 2,  1); set (1, 2, -2);        set (2, 2,  1);
        //divisor (1);
    }

    /** Evaluate the filter. */

    Color eval (float x, float y)
    {
        if (digCommonInput === null) return Color.Black;

        float r = 0, g = 0, b = 0, a = 0;
        float *m = digCommonMatrix;

        for (int py; py < digCommonSize; py ++)
        {
            for (int px; px < digCommonSize; px ++, m ++)
            {
                if (*m == 0) continue;

                float rx = x - digCommonCX + px;
                float ry = y - digCommonCY + py;

                contort.eval (rx, ry);
                Color c = digCommonInput.eval (rx, ry);

                r += (float) c.r * *m;
                g += (float) c.g * *m;
                b += (float) c.b * *m;
                a += (float) c.a * *m;
            }
        }

        Color result = SColor (r / digCommonDivisor, g / digCommonDivisor, b / digCommonDivisor, a / digCommonDivisor);
        return result;
    }

/+
#ifdef DoxygenShouldSkipThis
+/

    Brush digCommonInput; /* Input brush */
    float [] digCommonMatrix; /* Matrix */
    int digCommonSize; /* Size of the matrix */
    int digCommonCX, digCommonCY; /* Center */
    float digCommonDivisor; /* Divisor */
    bit digCommonUseDivisor; /* Divisor has been set by the user */
    
/+
#endif
+/
}
